let map, marker;

// 🔹 Botão "Rota no mapa" redesenha rota no Leaflet
function montarMapaEntregaComEndereco(lat, lng, elemento, enderecoTexto, revisao = false) {
    const container = document.getElementById(elemento);
    let button = '';
    if (!container) return;

    // console.log("Revisão:", revisao);

    if (revisao){
        button = `<button class="btn-custom" onclick="abrirModalEndereco();">
            <i class="fa-solid fa-pen-to-square"></i> Alterar
        </button>`;
    }
    else{
        button = ``;
    }

    container.innerHTML = `
    ${button}  
    <div class='bloco-loja' data-loja='$slug' style='cursor:pointer; border: 1px solid #ccc; padding: 10px; border-radius: 8px; background: #f9f9f9; transition: 0.3s;'>
        <div style="margin: 0; font-size: 12px;">
            ${enderecoTexto}
        </div>
        <div class='mapa-loja' style='margin-top: 10px;'>
            <iframe width='100%' height='100' style='border:0' loading='lazy'
                referrerpolicy="no-referrer-when-downgrade"
                src="https://maps.google.com/maps?q=${lat},${lng}&hl=pt&z=16&output=embed">
            </iframe>
        </div>
    </div>
    `;
}

// 🔹 Botão "Localização" obtem localização e redesenha rota no Leaflet
function getLocation() {
    navigator.geolocation.getCurrentPosition(success, error, {
        enableHighAccuracy: true,
        timeout: 10000,
        maximumAge: 0
    });

    function success(position) {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        atualizarMapa(lat, lng, true);
    }

    function error(err) {
        let message = "Erro ao obter localização: ";
        switch (err.code) {
            case err.PERMISSION_DENIED: message += "Permissão negada."; break;
            case err.POSITION_UNAVAILABLE: message += "Informações indisponíveis."; break;
            case err.TIMEOUT: message += "Tempo excedido."; break;
            default: message += err.message;
        }

        Swal.fire('Erro', message, 'error');
    }
}

// 🔹 Busca endereço com base na localização
// 🔹 Busca endereço com base na localização
function atualizarMapa(lat, lng, primeiraVez = false) {
    // Nominatim (reverse geocoding)
    const url = `https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${lat}&lon=${lng}&addressdetails=1&accept-language=pt-BR`;

    fetch(url)
        .then(res => res.json())
        .then(data => {
            const endereco = data?.display_name || "Endereço não encontrado.";
            const comp = data?.address || {};

            const cidade = comp.city || comp.town || comp.village || '';
            const bairro = comp.suburb || comp.neighbourhood || '';
            const rua = comp.road || '';
            const cep = comp.postcode || '';
            const estado = comp.state || '';
            const complemento = comp.building || comp.unit || comp.block || '';

            // Atualiza campos ocultos
            document.getElementById("formLatitude").value = lat;
            document.getElementById("formLongitude").value = lng;
            const campoNumero = document.getElementById("numeroEntrega");
            const numero = campoNumero?.value || '';
            const enderecoCompleto = numero ? `${endereco}, Nº ${numero}` : endereco;
            document.getElementById("formEndereco").value = enderecoCompleto;

            // Atualiza campos visíveis
            document.getElementById("cidadeEntrega").value = cidade;
            document.getElementById("bairroEntrega").value = bairro;
            document.getElementById("ruaEntrega").value = rua;
            document.getElementById("estadoEntrega").value = estado || '';

            const campoCep = document.getElementById("cepEntrega");
            if (cep) {
                campoCep.value = cep;
            } else {
                campoCep.value = '';
                Swal.fire('CEP não encontrado automaticamente', 'Preencha manualmente o campo de CEP.', 'info');
            }

            if (primeiraVez) {
                campoNumero.value = '';
            }

            const campoObs = document.querySelector('[name="observacao"], [name="complemento"]');
            if (campoObs) campoObs.value = complemento;
        })
        .catch(() => {
            Swal.fire('Erro', 'Erro ao buscar endereço com base na localização.', 'error');
        });

    // Mapa
    if (!map) {
        map = L.map('map').setView([lat, lng], 17);
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            maxZoom: 19
        }).addTo(map);

        marker = L.marker([lat, lng], { draggable: true }).addTo(map)
            .bindPopup('📍 Você está aqui').openPopup();

        marker.on('dragend', function (e) {
            const newLatLng = e.target.getLatLng();
            atualizarMapa(newLatLng.lat, newLatLng.lng, true);
        });

        setTimeout(() => map.invalidateSize(), 200);
    } else {
        map.setView([lat, lng], 17);
        marker.setLatLng([lat, lng]);
    }
}

// 🔹 Busca endereço com base nos campos preenchidos
function buscarEnderecoDoFormulario(primeiraVez = null, silencioso = false) {
  const cep     = document.getElementById('cepEntrega')?.value.trim() || '';
  const cidade  = document.getElementById('cidadeEntrega')?.value.trim() || '';
  const bairro  = document.getElementById('bairroEntrega')?.value.trim() || '';
  const rua     = document.getElementById('ruaEntrega')?.value.trim() || '';
  const numero  = document.getElementById('numeroEntrega')?.value.trim() || '';
  const estado  = document.getElementById('estadoEntrega')?.value.trim() || '';

  const temMinimo = rua && bairro && cidade;
  if (!temMinimo) {
    if (!silencioso) {
      Swal.fire('Atenção', 'Preencha Rua, Bairro e Cidade para buscar o endereço.', 'warning');
    }
    return;
  }

  const partes = [rua, numero, bairro, cidade, estado, cep].filter(Boolean);
  const enderecoCompleto = partes.join(', ');

  const url = `https://nominatim.openstreetmap.org/search?format=jsonv2&q=${encodeURIComponent(enderecoCompleto)}&limit=1&addressdetails=1&accept-language=pt-BR&countrycodes=br`;

  fetch(url)
    .then(res => res.json())
    .then(data => {
      if (!Array.isArray(data) || data.length === 0) {
        if (!silencioso) {
          Swal.fire('Endereço não encontrado', 'Não conseguimos localizar o endereço informado.', 'error');
        }
        return;
      }
      const lat = parseFloat(data[0].lat);
      const lng = parseFloat(data[0].lon);
      atualizarMapa(lat, lng, !!primeiraVez /* primeiraVez */);
    })
    .catch(() => {
      if (!silencioso) {
        Swal.fire('Erro', 'Erro ao buscar localização geográfica.', 'error');
      }
    });
}

buscarEnderecoDoFormulario(false, true); // silencioso

// 🔹 Busca endereço com base no CEP
document.getElementById('cepEntrega').addEventListener('blur', () => {
    const cep = document.getElementById('cepEntrega').value.replace(/\D/g, '');
    if (cep.length >= 8) {
        fetch(`https://viacep.com.br/ws/${cep}/json/`)
            .then(res => res.json())
            .then(data => {
                if (data.erro) {
                    Swal.fire('CEP inválido', 'CEP não encontrado nos Correios.', 'warning');
                    return;
                }

                document.getElementById('cidadeEntrega').value = data.localidade || '';
                document.getElementById('bairroEntrega').value = data.bairro || '';
                document.getElementById('ruaEntrega').value = data.logradouro || '';
                document.getElementById('estadoEntrega').value = data.uf || '';
                buscarEnderecoDoFormulario(true, true); // silencioso ao completar via CEP
            })
            .catch(() => {
                Swal.fire('Erro', 'Erro ao buscar o CEP.', 'error');
            });
    }
});

// Mostra o mapa de entrega
function montarMapaEntrega(){
    const $sel = $('input[name="opcao-endereco"]:checked');
    const d = $sel.data() || {};

    const cep    = (d.cep ?? '').toString().trim();
    const cidade = (d.cidade ?? '').toString().trim();
    const bairro = (d.bairro ?? '').toString().trim();
    const rua    = (d.rua ?? '').toString().trim();
    const numero = (d.numero ?? '').toString().trim();
    const estado = (d.estado ?? '').toString().trim();
    const complemento = (d.complemento ?? '').toString().trim();

    const lat = Number.isFinite(parseFloat(d.lat)) ? parseFloat(d.lat) : NaN;
    const lng = Number.isFinite(parseFloat(d.lng)) ? parseFloat(d.lng) : NaN;

    if(complemento){
        text = ' - ' + complemento;
    }
    else{
        text = '';
    }


    montarMapaEntregaComEndereco(lat, lng, 'loja-entrega',`${rua}, ${numero} - ${bairro}, ${cidade}/${estado}, ${cep} ${text}`);
}