<?php
/**
 * server-processing-complete-profile.php
 * Completa o cadastro (UPDATE para usuário existente ou INSERT para novo vindo do Google).
 */
session_start();
require_once $_SERVER['DOCUMENT_ROOT'] . '/wp-load.php';
header('Content-Type: application/json; charset=utf-8');

global $database_connection;
if (!$database_connection || !($database_connection instanceof mysqli)) {
  echo json_encode(['success' => false, 'message' => 'Sem conexão com o banco.']); exit;
}

/* ===================== Helpers ===================== */

function fail($msg){ echo json_encode(['success'=>false,'message'=>$msg], JSON_UNESCAPED_UNICODE); exit; }

/** valida CPF (com ou sem máscara) */
function validateCPF_local($cpf){
  $cpf = preg_replace('/\D/','', $cpf);
  if (strlen($cpf) != 11) return false;
  if (preg_match('/(\d)\1{10}/', $cpf)) return false;
  for ($i=0,$j=10,$s=0;$i<9;$i++,$j--) $s += $cpf[$i]*$j; $r=$s%11; $dv1 = ($r<2)?0:11-$r;
  for ($i=0,$j=11,$s=0;$i<10;$i++,$j--) $s += $cpf[$i]*$j; $r=$s%11; $dv2 = ($r<2)?0:11-$r;
  return ($cpf[9]==$dv1 && $cpf[10]==$dv2);
}

/** checa duplicidade de CPF (ignora máscara), com opção de excluir um cliente_uuid */
function cpf_exists_other($cpf_digits, mysqli $conn, $exclude_uuid=null){
  $sql = "SELECT 1 FROM customers
          WHERE REPLACE(REPLACE(REPLACE(cpf,'.',''),'-',''),' ','') = ?" .
          ($exclude_uuid ? " AND cliente_uuid <> ?" : "") . " LIMIT 1";
  $st = $conn->prepare($sql);
  if ($exclude_uuid) { $st->bind_param('ss', $cpf_digits, $exclude_uuid); }
  else { $st->bind_param('s', $cpf_digits); }
  $st->execute();
  $r = $st->get_result();
  $exists = ($r && $r->num_rows > 0);
  $st->close();
  return $exists;
}

/** distância em KM */
function calcular_distancia($lat1,$lon1,$lat2,$lon2){
  $R=6371; $dLat=deg2rad($lat2-$lat1); $dLon=deg2rad($lon2-$lon1);
  $a=sin($dLat/2)**2 + cos(deg2rad($lat1))*cos(deg2rad($lat2))*sin($dLon/2)**2;
  return 2*$R*atan2(sqrt($a),sqrt(1-$a));
}

/** coluna existe? */
function col_exists(mysqli $c,$table,$col){
  $st = $c->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  if(!$st) return false;
  $st->bind_param('s',$col); $st->execute();
  $r = $st->get_result(); $ok = $r && $r->num_rows>0;
  $st->close(); return $ok;
}

/* ===================== Entrada ===================== */

$d          = $_POST;
$cpf        = trim($d['cpf'] ?? '');
$celular    = trim($d['celular'] ?? '');
$nasc_br    = trim($d['nascimento'] ?? '');
$genero     = trim($d['genero'] ?? 'N/F');
$cep        = trim($d['cep'] ?? '');
$estado     = trim($d['estado'] ?? '');
$cidade     = trim($d['cidade'] ?? '');
$bairro     = trim($d['bairro'] ?? '');
$rua        = trim($d['rua'] ?? '');
$numero     = trim($d['numero'] ?? '');

/* ===================== Validações ===================== */

if (!validateCPF_local($cpf)) fail('CPF inválido.');

$cpf_digits = preg_replace('/\D/','', $cpf); // p/ integrações
// grava no banco FORMATADO (000.000.000-00)
$cpf_num = (strlen($cpf_digits)===11)
  ? substr($cpf_digits,0,3).'.'.substr($cpf_digits,3,3).'.'.substr($cpf_digits,6,3).'-'.substr($cpf_digits,9,2)
  : $cpf;

$required = ['celular','cep','estado','cidade','bairro','rua','numero','genero'];
foreach ($required as $f) {
  if (empty($$f)) fail("Campo obrigatório ausente: $f");
}

// nascimento dd/mm/aaaa -> yyyy-mm-dd
$data_nasc = '0000-00-00';
if ($nasc_br) {
  @list($dia,$mes,$ano) = explode('/', $nasc_br);
  if (!checkdate(intval($mes), intval($dia), intval($ano))) fail('Data de nascimento inválida.');
  $data_nasc = sprintf('%04d-%02d-%02d',$ano,$mes,$dia);
}

/* ===================== Geo e loja mais próxima ===================== */

$lat_long = '';
if (function_exists('get_lat_long')) {
  $lat_long = get_lat_long($rua, $numero, $bairro, $cidade, $estado, $cep);
}
$lat_u = $lng_u = null;
if ($lat_long && strpos($lat_long, ',') !== false) {
  @list($lat_u, $lng_u) = array_map('floatval', explode(',', $lat_long));
}

$loja_mais_proxima = null;
try {
  $lojas = get_terms(['taxonomy'=>'loja','hide_empty'=>false]);
  if ($lojas && !is_wp_error($lojas)) {
    $best = PHP_FLOAT_MAX;
    foreach ($lojas as $loja) {
      $coords = function_exists('get_field') ? get_field('coordenadas', $loja) : '';
      if (!$coords) continue;
      @list($lat_l,$lng_l) = array_map('floatval', explode(',', $coords));
      if (is_numeric($lat_u)&&is_numeric($lng_u)&&is_numeric($lat_l)&&is_numeric($lng_l)) {
        $d = calcular_distancia($lat_u,$lng_u,$lat_l,$lng_l);
        if ($d < $best) { $best=$d; $loja_mais_proxima = $loja->term_id; }
      } elseif ($loja_mais_proxima === null) {
        $loja_mais_proxima = $loja->term_id;
      }
    }
  }
} catch (Throwable $e) { /* ignora */ }

// limpar número
$numero_limpo = preg_replace('/\D/','', $numero);

/* ===================== UPDATE (usuário já existe) ===================== */
if (!empty($_SESSION['usuario']['id'])) {
  $userID = $_SESSION['usuario']['id'];

  if (cpf_exists_other($cpf_digits, $database_connection, $userID)) {
    fail('CPF já cadastrado.');
  }

  $sql = "UPDATE customers SET 
            cpf=?, celular=?, data_nascimento=?, genero=?,
            cep=?, estado=?, cidade=?, bairro=?, rua=?, numero=?, coordenadas=?," .
            ($loja_mais_proxima ? "loja_mais_frequentada=?," : "") .
            " data = NOW()
          WHERE cliente_uuid = ?";

  if ($loja_mais_proxima) {
    $st = $database_connection->prepare($sql);
    $st->bind_param(
      'sssssssssssis',
      $cpf_num, $celular, $data_nasc, $genero,
      $cep, $estado, $cidade, $bairro, $rua, $numero_limpo,
      $lat_long, $loja_mais_proxima, $userID
    );
  } else {
    $sql = str_replace("loja_mais_frequentada=?,", "", $sql);
    $st = $database_connection->prepare($sql);
    $st->bind_param(
      'ssssssssssss', // 12 tipos para 12 valores
      $cpf_num, $celular, $data_nasc, $genero,
      $cep, $estado, $cidade, $bairro, $rua, $numero_limpo,
      $lat_long, $userID
    );
  }

  if (!$st || !$st->execute()) fail('Erro ao atualizar cadastro.');
  $st->close();

  // integração/PDV (se existir)
  try {
    if (function_exists('get_status_api') && get_status_api(10)==1 && function_exists('save_user_api')) {
      $nome  = function_exists('get_info') ? get_info('nome', $userID)  : '';
      $email = function_exists('get_info') ? get_info('email', $userID) : '';
      $ret   = save_user_api($nome,$cpf_digits,$rua,$numero_limpo,$bairro,$cidade,$estado,$cep,$celular,$email,$genero);

      $_sql = "UPDATE customers SET pdv=? WHERE cliente_uuid=?";
      $_st  = $database_connection->prepare($_sql);
      $pdv  = ($ret==1)?'1':'0';
      $_st->bind_param('ss', $pdv, $userID); $_st->execute(); $_st->close();

      if ($ret==1 && function_exists('statusEntidade')) statusEntidade($userID, "S");
    }
  } catch (Throwable $t) { /* ignora */ }

  echo json_encode(['success'=>true]); exit;
}

/* ===================== INSERT (novo — Google em sessão) ===================== */
if (empty($_SESSION['google_onboarding'])) {
  fail('Sessão expirada. Faça login com o Google novamente.');
}
$g = $_SESSION['google_onboarding'];
$nome   = $g['name']      ?? '';
$email  = $g['email']     ?? '';
$gid    = $g['google_id'] ?? null;
$verif  = !empty($g['verified']) ? 1 : 0;
$pic    = $g['picture']   ?? '';

if (cpf_exists_other($cpf_digits, $database_connection, null)) {
  fail('CPF já cadastrado.');
}

/* assinatura / pdv (mesma lógica do seu register) */
$dataAtual = date('d-m-Y');
if (function_exists('get_field') && get_field('cobrar_assinatura', 'options') == true) {
  if (get_field('ativacao_padrao', 'options') == 'ativos') {
    $diasPadrao     = get_field('dias_padrao_ativacao', 'options');
    $dataVencimento = date('d-m-Y', strtotime($dataAtual . ' + ' . $diasPadrao . ' days'));
    $statusPDV      = "S";
  } else {
    $dataVencimento = $dataAtual;
    $statusPDV      = "N";
  }
} else {
  $statusPDV = "S";
  $dataVencimento = $dataAtual;
}

$senhaRand = md5(bin2hex(random_bytes(8)));
$credito   = '0,00';
$indicador = 0;
$loja      = $loja_mais_proxima ?: '';
$pdv_inicial = '0';

/* === Regras de comunidade / grupo WhatsApp (iguais ao register) === */
$community = function_exists('get_data') ? get_data('admin_config', 'community_required') : 0;
$ativo = $community ? 0 : 1;     // se comunidade é obrigatória, entra inativo até confirmar
$wpp_group_status = 0;

// Formata telefone para 55DDxxxxxxxxx (mesma regex do register)
$telefone_lookup = preg_replace('/^\(?(\d{2})\)?\s*(\d{4,5})-?\s*(\d{4})$/', '55$1$2$3', $celular);

// Se já houver relação de grupo (status = 2 "pré-vinculado"), ativa direto e marca wpp_group_status=1
$id_related_group = null;
if ($stmt = $database_connection->prepare("SELECT id FROM wpp_groups_related WHERE customer_id = ? AND status = 2 LIMIT 1")) {
  $stmt->bind_param('s', $telefone_lookup);
  $stmt->execute();
  $res = $stmt->get_result();
  if ($res && $res->num_rows > 0) {
    $row = $res->fetch_assoc();
    $id_related_group = $row['id'];
    $wpp_group_status = 1;
    $ativo = 1;
  }
  $stmt->close();
}

/* INSERT via sua função existente (27 parâmetros) */
$userID = insert_customers(
  $database_connection, '', $nome, $cpf_num, $celular, $email,
  $data_nasc, $genero, $cep, $estado, $cidade, $bairro, $rua, (int)$numero_limpo,
  $senhaRand, $credito, $indicador, $loja, '0', $dataVencimento,
  $lat_long, '0000-00-00 00:00:00', date('Y-m-d H:i:s'), 'publish', $pdv_inicial,
  $ativo, $wpp_group_status
);
if (!$userID) fail('Erro ao inserir cadastro.');

/* Finaliza vínculo de grupo se houver pré-relação */
if ($wpp_group_status == 1 && $id_related_group) {
  if ($up = $database_connection->prepare("UPDATE wpp_groups_related SET customer_id = ?, status = 1 WHERE id = ?")) {
    $up->bind_param('si', $userID, $id_related_group);
    $up->execute();
    $up->close();
  }
}

/* grava metadados do Google se colunas existirem */
try {
  $sets=[]; $types=''; $vals=[];
  if (col_exists($database_connection,'customers','google_id'))      { $sets[]='google_id=?';      $types.='s'; $vals[]=$gid; }
  if (col_exists($database_connection,'customers','google_verified')){ $sets[]='google_verified=?';$types.='i'; $vals[]=$verif; }
  if (col_exists($database_connection,'customers','picture'))        { $sets[]='picture=?';        $types.='s'; $vals[]=$pic; }
  if (col_exists($database_connection,'customers','provider'))       { $sets[]='provider=?';       $types.='s'; $vals[]='google'; }
  if ($sets){
    $sql="UPDATE customers SET ".implode(',', $sets)." WHERE cliente_uuid=?";
    $types.='s'; $vals[]=$userID;
    $up=$database_connection->prepare($sql);
    if ($up){ $up->bind_param($types, ...$vals); $up->execute(); $up->close(); }
  }
} catch (Throwable $t) {}

/* integração PDV */
try {
  if (function_exists('get_status_api') && get_status_api(10)==1 && function_exists('save_user_api')) {
    $ret = save_user_api($nome,$cpf_digits,$rua,$numero_limpo,$bairro,$cidade,$estado,$cep,$celular,$email,$genero);

    $_sql = "UPDATE customers SET pdv=? WHERE cliente_uuid=?";
    $_st  = $database_connection->prepare($_sql);
    $pdv  = ($ret==1)?'1':'0';
    $_st->bind_param('ss', $pdv, $userID); $_st->execute(); $_st->close();

    if ($ret==1 && function_exists('statusEntidade')) statusEntidade($userID, $statusPDV);
  }
} catch (Throwable $t) {}

/* cria sessão de login e limpa onboarding */
$_SESSION['usuario'] = ['id' => $userID];
unset($_SESSION['google_onboarding']);

echo json_encode(['success'=>true]);